set.seed(1691)

#' Calculates AUC, confidence intervals, and generates a ROC plot.
#'
#' @param data A data frame containing at least two columns:
#'   \describe{
#'     \item{biomarker}{Numeric values representing the diagnostic marker.}
#'     \item{status}{Character or factor with levels `"0"` (controls) and `"1"` (cases).}
#'   }
#'
#' @param method A character string specifying the ROC/AUC modeling approach.
#'   Supported options include:
#'   \itemize{
#'     \item `"empirical"` – empirical ROC
#'     \item `"order"` – ROC curve under stochastic order constraints
#'     \item `"norm_silver"` – kernel ROC with normal kernel and Silverman bandwidth
#'     \item `"norm_ucv"` – kernel ROC with normal kernel and UCV bandwidth
#'     \item `"bi_silver"` – kernel ROC with biweight kernel and Silverman bandwidth
#'     \item `"bi_ucv"` – kernel ROC with biweight kernel and UCV bandwidth
#'     \item `"binormal"` – classical binormal ROC model
#'     \item `"biweibull"` – parametric bi-Weibull ROC
#'     \item `"bigamma"` – parametric ROC assuming gamma distributions
#'     \item `"lehmann"` – ROC under the Lehmann alternative
#'     \item `"bayesbiweibull"` – Bayesian bi-Weibull ROC (MCMC-based)
#'     \item `"BB"` – Bayesian bootstrap ROC
#'     \item `"dpm"` – Dirichlet process mixture ROC
#'   }
#'
#' @param ci Logical; if `TRUE` (default), computes confidence intervals for the AUC
#'   (or credible intervals for Bayesian methods).
#'
#' @param ci_method Character string specifying the type of interval estimation.
#'   Not all CI methods are compatible with every model:
#'   \itemize{
#'     \item `"delong"` – DeLong’s variance-based normal approximation
#'     \item `"bootstrap"` – nonparametric bootstrap interval
#'     \item `"hm"` – Hanley–McNeil variance-based interval
#'     \item `"mle"` – likelihood-based interval
#'     \item `"all"` – computes all applicable interval types for the selected method
#'   }
#'
#' @param siglevel Numeric; significance level \eqn{\alpha} for the confidence interval.
#'   The corresponding confidence level is \eqn{1 - \alpha}.
#'
#' @param boot_iter Integer; number of bootstrap resamples (used when
#'   `ci_method = "bootstrap"` or `"all"`). Larger values give more stable intervals
#'   but increase computation time.
#'
#' @return A list with the following elements:
#'   \describe{
#'     \item{summary}{Printed output of the AUC and confidence intervals.}
#'     \item{plot}{A `ggplot` object visualizing the ROC curve.}
#'   }
#'   The exact structure may vary depending on the chosen model.
#'
#' @import ggplot2
#' @import kedd
#' @import dplyr
#' @import survival
#' @import nleqslv
#' @import HDInterval
#' @import ROCit
#' @import doParallel
#' @import foreach
#' @import nor1mix
#' @import pbivnorm
#' @import parallel
#' @import readr
#' @examples
#' # Import well formated dataset
#' data(DMDmodified) 
#' # Calculate AUC summary and ROC plot
#' auc <- AUC(
#'   data=DMDmodified,
#'   method = "empirical",
#'   ci        = TRUE
#' )
#' # Get the AUC summary
#' cat(auc$summary)
#' # Get the ROC plot
#' auc$plot
#' @export

#' @importFrom stats IQR dgamma dlnorm dnorm na.omit pbeta pgamma pnorm qnorm
#' @importFrom stats quantile rbeta rexp rgamma rnorm runif rweibull sd var
#' @importFrom MASS fitdistr

AUC <- function(data, method, ci = TRUE, ci_method = "delong", siglevel = 0.05, boot_iter = 1000) {
  df <- na.omit(data)
  pos <- df[df$status == "1",]
  neg <- df[df$status == "0",]
  X <- neg$biomarker
  Y <- pos$biomarker
  m <- length(X)
  n <- length(Y)

  if(method == "norm_silver"){
    norm_silver(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "norm_ucv"){
    norm_ucv(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "bi_silver"){
    bi_silver(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "bi_ucv"){
    bi_ucv(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "order"){
    master_order(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "empirical"){
    master_empirical(X, Y, m, n, ci, ci_method, siglevel, boot_iter)
  }else if(method == "bayesbiweibull"){
    set.seed(1691)
    calculate_auc_ci_plot_beweibull(X,Y,m,n,ci,siglevel)
  }else if(method == "BB"){
    bb(X,Y,ci,siglevel,boot_iter)
  }else if(method == "dpm"){
    master_dpm(X,Y,ci,siglevel)
  }else if(method == "bigamma"){
    gamma_weibull(X, Y, ci,siglevel,boot_iter)
  }else if(method == "binormal"){
    if(ci_method == "delong"){
      ci_method = "bootstrap"
    }
    binormal(X,Y,ci,ci_method,siglevel,boot_iter)
  }else if(method == "biweibull"){
    if(ci_method == "delong"){
      ci_method = "mle"
    }
    master_biweibull(X,Y,ci,ci_method,siglevel,boot_iter)
  }else if(method =="lehmann"){
    lehmann(X,Y,ci,ci_method,siglevel)
  }
}

norm_silver <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  h_x <- bandwidth_norm_silver(X)
  h_y <- bandwidth_norm_silver(Y)

  roc_data <- ROC_kde_normal_norm_silver(X, Y, h_x, h_y)
  auc_kde_normal <- calculate_auc_norm_silver(roc_data$FPR, roc_data$TPR)

  # Output results for empirical AUC
  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC estimate :",round(auc_kde_normal,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_norm_silver(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_norm_silver(X, Y)
      ci_delong <- calc_ci_auc_delong_norm_silver(auc = auc_kde_normal, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% DeLong CI :","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
  }

  ##Print plot

  # Bandwidths
  h_silver_x <- bandwidth(X)
  h_silver_y <- bandwidth(Y)

  # Compute ROC curve
  roc_kde_normal_silver   <- ROC_kde(X, Y, h_silver_x, h_silver_y, normal_kernel)

  # Label
  roc_kde_normal_silver$Method   <- "KDE (Normal, SM)"

  roc_data <- roc_kde_normal_silver


  # Plot with ggplot
  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c(
      "KDE (Normal, SM)" = "purple"
    )) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(legend.position = "none",
          panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

norm_ucv <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  h_x <- h.ucv(X)$h
  h_y <- h.ucv(Y)$h

  roc_data <- ROC_kde_normal_norm_ucv(X, Y, h_x, h_y)
  auc_kde_normal <- calculate_auc_norm_ucv(roc_data$FPR, roc_data$TPR)

  # Output results for empirical AUC
  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC estimate :",round(auc_kde_normal,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_norm_ucv(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_norm_ucv(X, Y)
      ci_delong <- calc_ci_auc_delong_norm_ucv(auc = auc_kde_normal, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% DeLong CI :","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
  }

  h_ucv_x <- h.ucv(X)$h
  h_ucv_y <- h.ucv(Y)$h

  # Compute all 4 ROC curves
  roc_kde_normal_ucv      <- ROC_kde(X, Y, h_ucv_x, h_ucv_y, normal_kernel)

  roc_kde_normal_ucv$Method      <- "KDE (Normal, UCV)"

  roc_data <- roc_kde_normal_ucv

  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c(
      "KDE (Normal, UCV)" = "orange"
    )) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(legend.position = "none",
          panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))  
}

bi_silver <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  h_x <- bandwidth_bi_silver(X)
  h_y <- bandwidth_bi_silver(Y)

  roc_data <- ROC_kde_biweight_bi_silver(X, Y, h_x, h_y)
  auc_kde_biweight <- calculate_auc_bi_silver(roc_data$FPR, roc_data$TPR)

  # Output results for empirical AUC
  msg <-"Summary:\n"
  msg <- paste0(msg,"AUC estimate :",round(auc_kde_biweight,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_bi_silver(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_bi_silver(X, Y)
      ci_delong <- calc_ci_auc_delong_bi_silver(auc = auc_kde_biweight, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% DeLong CI :","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
  }

  ##Print plot

  # Bandwidths
  h_silver_x <- bandwidth_bi_silver(X)
  h_silver_y <- bandwidth_bi_silver(Y)

  # Compute ROC curve
  roc_kde_biweight_silver <- ROC_kde(X, Y, h_silver_x, h_silver_y, biweight_kernel)

  # Label
  roc_kde_biweight_silver$Method <- "KDE (Biweight, SM)"

  roc_data <- roc_kde_biweight_silver


  # Plot with ggplot
  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c(
      "KDE (Biweight, SM)" = "black"
    )) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(legend.position = "none",
          panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

bi_ucv <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  h_x <- h.ucv(X)$h
  h_y <- h.ucv(Y)$h

  roc_data <- ROC_kde_biweight_bi_ucv(X, Y, h_x, h_y)
  auc_kde_biweight <- calculate_auc_bi_ucv(roc_data$FPR, roc_data$TPR)

  # Output results for empirical AUC
  msg <- "Summary\n"
  msg <- paste0(msg,"AUC estimate :",round(auc_kde_biweight,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_bi_ucv(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_bi_ucv(X, Y)
      ci_delong <- calc_ci_auc_delong_bi_ucv(auc = auc_kde_biweight, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% DeLong CI :","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
  }

  h_ucv_x <- h.ucv(X)$h
  h_ucv_y <- h.ucv(Y)$h

  # Compute all 4 ROC curves
  roc_kde_biweight_ucv    <- ROC_kde(X, Y, h_ucv_x, h_ucv_y, biweight_kernel)

  roc_kde_biweight_ucv$Method    <- "KDE (Biweight, UCV)"

  roc_data <- roc_kde_biweight_ucv

  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c(
      "KDE (Biweight, UCV)" = "brown"
    )) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(legend.position = "none",
          panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))

}

master_order <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  auc_hat_emp <- calc_auc_empirical_order(X = X, Y = Y)
  roc_data_order <- roc_order(X, Y)
  auc_hat_order <- calculate_auc_order(roc_data_order$FPR, roc_data_order$TPR)
  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC estimate :",round(auc_hat_order,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_order(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg, (1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_order(X = X, Y = Y)
      ci_delong <- calc_ci_auc_delong_order(auc = auc_hat_order, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% DeLong CI :","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
    if(ci_method == "hm" | ci_method == "all") {
      var_auc_emp_hm <- calc_var_auc_empirical_order(auc = auc_hat_emp, X = X, Y = Y)
      order_hm <- calc_ci_auc_hm_order(auc = auc_hat_order, var_auc = var_auc_emp_hm, alpha = siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% HM CI :","(", round(order_hm[1],5),",",round(order_hm[2],5),")", "\n")
    }
  }

  roc_data <- roc_order(X, Y)

  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Order" = "blue")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))

}

master_empirical <- function(X, Y, m, n, ci=TRUE,ci_method="delong",siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  auc_hat_emp <- calc_auc_empirical(X = X, Y = Y)
  msg <- "Summary:\n"
  msg <- paste0(msg, "AUC estimate :",round(auc_hat_emp,5), "\n")

  if(ci == TRUE){
    if(ci_method == "bootstrap" | ci_method == "all"){
      boot_ci <- bootstrap_auc_ci_empirical(X, Y, boot_iter, 1-siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% Bootstrap CI :","(", round(boot_ci[1],5),",",round(boot_ci[2],5),")", "\n")
    }
    if(ci_method == "delong" | ci_method == "all") {
      var_auc_emp_delong <- calc_var_auc_delong_empirical(X = X, Y = Y)
      ci_delong <- calc_ci_auc_delong_empirical(auc = auc_hat_emp, var_auc = var_auc_emp_delong, alpha = siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% DeLong CI: ","(", round(ci_delong[1],5),",",round(ci_delong[2],5),")", "\n")
    }
    if(ci_method == "hm" | ci_method == "all") {
      var_auc_emp_hm <- calc_var_auc_empirical_order(auc = auc_hat_emp, X = X, Y = Y)
      ci_hm <- calc_ci_auc_hm_empirical(auc = auc_hat_emp, var_auc = var_auc_emp_hm, alpha = siglevel)
      msg <- paste0(msg,(1-siglevel)*100,"% HM CI :","(", round(ci_hm[1],5),",",round(ci_hm[2],5),")", "\n")
    }
  }

  roc_data <- roc_empirical(X, Y)
  roc_data_modified <- roc_data %>% filter(!(FPR %in% c(0,1))) %>%
                                 add_row(FPR = 0, TPR = 0,Method = "Empirical")
  roc_data_modified <- roc_data_modified %>%
                                add_row(FPR = 1, TPR = 1,Method = "Empirical")

  p <- ggplot(roc_data_modified, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Empirical" = "darkgreen")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

bb <- function(X,Y,ci=TRUE,siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  t_values <- seq(0, 1, by = 0.01)

  # Compute the BB estimate of the ROC curve
  ROC_bb <- ROC_realization_bb(X, Y, t_values, boot_iter)

  AUC_k <- rep(NA, 1000)
  for (i in 1:length(AUC_k)) {
    AUC_k[i] <- calculate_AUC_bb(ROC_bb[i, ], t_values)$V1
  }
  auc_bb <- mean(AUC_k)

  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC estimate :", round(auc_bb,5), "\n")

  if(ci == TRUE){
    ci_bb<- credibleble_interval_bb(AUC_k,siglevel)
    msg <- paste0(msg,(1-siglevel)*100, "% CI : ","(", round(ci_bb[1],5),",",round(ci_bb[2],5),")","\n")
  }

  # Compute the mean and confidence intervals of the ROC ensemble
  ROC_mean <- colMeans(ROC_bb)  # Average ROC curve

  roc_bb <- data.frame(FPR = t_values, TPR = ROC_mean, Method = "BB")

  p <- ggplot(roc_bb, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1.2) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("BB" = "magenta")) +
    labs(title = "ROC Curves: Bayesian Bootstrap",
         x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(
      legend.position = "none",
      panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

master_dpm <- function(x, y, ci=TRUE,siglevel=0.05){
  # 4. Run the analysis specifying priors
  prior <- list(m0 = 0, S0 = 0.1, a = 0.1, b = 0.1, alpha = 1, L = 10)
  mcmc <- list(nsave = 400, nburn = 100, nskip = 1)

  set.seed(1691)
  fit_x <- dpm(y = x, prior = prior, mcmc = mcmc, standardise = TRUE)
  fit_y <- dpm(y = y, prior = prior, mcmc = mcmc, standardise = TRUE)

  p_grid <- seq(0, 1, length.out = 101)
  roc_samples <- compute_roc_dpm(fit_x, fit_y, p_grid)
  auc_samples <- compute_auc_dpm(fit_x, fit_y)

  # 5. Calculate statistics and plot
  roc_mean <- colMeans(roc_samples)
  roc_ci <- apply(roc_samples, 2, quantile, c(0.025, 0.975))
  auc_mean <- mean(auc_samples)
  auc_ci <- credibleble_interval_dpm(auc_samples, siglevel)

  msg <- "Summary:\n"
  msg <- paste0("AUC (Posterior Mean):", round(auc_mean, 5), "\n")
  if(ci == TRUE){
    msg <- paste0(msg, (1-siglevel)*100, "% Credible Interval:","(", round(auc_ci[1],5),",",round(auc_ci[2],5),")", "\n")
  }

  roc_data <- data.frame(
    FPR = p_grid,
    TPR = roc_mean,
    Method = "DPM"
  )

  p <- ggplot(roc_data, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1.2) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("DPM" = "orchid4")) +
    labs(title = "ROC Curves: DPM",
         x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(
      legend.position = "none",
      panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

master_biweibull <- function(X,Y,ci, ci_method,siglevel=0.05,boot_iter=1000){
  set.seed(1691)
  # Get the number of observations in each group
  m <- length(X)  # Number of not recovered observations
  n <- length(Y)  # Number of recovered observations

  para_est <- estimate_param_biweibull(X, Y)
  a_hat <- para_est$alpha
  s0_hat <- para_est$s0  # s0 is theta_0
  s1_hat <- para_est$s1  # s1 is theta_1

  # Calculate the AUC estimate using AUC = theta_1_hat / (theta_1_hat + theta_0_hat)
  auc_hat <- s1_hat / (s1_hat + s0_hat)  # ML estimate of AUC

  auc_ci_mle_weibull <- auc_ci_mle_weibull(auc_hat, m,n,a_hat, s0_hat, s1_hat, 1-siglevel)
  auc_ci_boot_weibull <- bootstrap_auc_ci_biweibull(X, Y,1-siglevel,boot_iter)

  # Print the confidence interval
  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC :", round(auc_hat,5), "\n")
  if(ci){
    if(ci_method == "mle" | ci_method == "all"){
      msg <- paste0(msg, (1-siglevel)*100,"% CI MLE:","(", round(auc_ci_mle_weibull[1],5),",",round(auc_ci_mle_weibull[2],5),")", "\n")
    }
    if(ci_method == "bootstrap" | ci_method == "all"){
      msg <- paste0(msg,(1-siglevel)*100, "% CI Bootstrap:(", round(auc_ci_boot_weibull[1],5),",",round(auc_ci_boot_weibull[2],5),")", "\n")
    }
  }

  df_biweibull <- data.frame(
    FPR = seq(0, 1, length.out = 100),
    TPR = seq(0, 1, length.out = 100) ^ ((1 - auc_hat) / auc_hat),
    Method = "Bi-Weibull"
  )

  p <- ggplot(df_biweibull, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Bi-Weibull" = "darkblue")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) +
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )

  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

if (getRversion() >= "2.15.1")  utils::globalVariables(c(
  "FPR", "TPR", "Method", "i"
))

