
calculate_auc_ci_plot_beweibull <- function(x,y,m,n,ci=TRUE,siglevel){
  
  # MCMC parameters for each simulation
  M <- 11000     # Total iterations per simulation
  burnin <- 1000
  total_samples <- (M - burnin)
  
  # Gibbs sampling setup
  samples <- matrix(NA, nrow = total_samples, ncol = 4)
  colnames(samples) <- c("alpha", "theta0", "theta1", "AUC")
  
  #prior choice
  a0 = 0
  b0 = 0 
  a1 = 0
  b1 = 0
  k = 0.1
  beta = 1
  
  # Initialize parameters
  alpha <- rgamma(1, shape = k, rate = 1/beta)  # Prior sample
  theta0 <- mean(x^alpha)^(1/alpha)
  theta1 <- mean(y^alpha)^(1/alpha)
  
  # M-H tuning
  accept <- 0
  sigma_alpha <- 0.05
  
  # Gibbs sampler
  sample_counter <- 1
  for (t in 1:M) {
    # Sample theta0 and theta1
    theta0 <- 1 / rgamma(1, shape = a0+m, rate = b0+sum(x^alpha))
    theta1 <- 1 / rgamma(1, shape = a1+n, rate = b1+sum(y^alpha))
    
    # M-H step for alpha
    alpha_prop <- exp(rnorm(1, mean = log(alpha), sd = sigma_alpha))
    
    # Log-likelihoods
    log_prior_prop <- -log(alpha_prop)  # Gamma(0,1) prior
    log_prior_current <- dgamma(alpha, shape = 0.1, rate = 1, log = TRUE)
    
    log_lik_prop <- (m + n)*log(alpha_prop) - m*log(theta0) - n*log(theta1) +
      (alpha_prop - 1)*(sum(log(x)) + sum(log(y))) -
      sum(x^alpha_prop)/theta0 - sum(y^alpha_prop)/theta1
    
    log_lik_current <- (m + n)*log(alpha) - m*log(theta0) - n*log(theta1) +
      (alpha - 1)*(sum(log(x)) + sum(log(y))) -
      sum(x^alpha)/theta0 - sum(y^alpha)/theta1
    
    # Proposal densities
    log_q_prop <- dlnorm(alpha, meanlog = log(alpha_prop), sdlog = sigma_alpha, log = TRUE)
    log_q_current <- dlnorm(alpha_prop, meanlog = log(alpha), sdlog = sigma_alpha, log = TRUE)
    
    log_r <- (log_lik_prop + log_prior_prop - log_q_prop) - 
      (log_lik_current + log_prior_current - log_q_current)
    
    if(!is.na(log_r)){
      if (log(runif(1)) < log_r) {
        alpha <- alpha_prop
        if (t > burnin) accept <- accept + 1
      }
    
      if (t > burnin) {
        auc <- theta1 / (theta0 + theta1)
        samples[sample_counter, ] <- c(alpha, theta0, theta1, auc)
        sample_counter <- sample_counter + 1
      }
    }
  }
  
  # Results - now we have more samples (16,000)
  AUC_post <- samples[,"AUC"]
  msg <- "Summary:\n"
  msg <- paste0(msg, "Bayesian AUC Estimates:\n")
  msg <- paste0(msg, "Posterior Mean:", round(mean(AUC_post), 5), "\n")
  if(ci == TRUE){
    msg <- paste0(msg, "95% HPD Interval:", "(",round(hdi(AUC_post,1-siglevel)[1],5),",",round(hdi(AUC_post,1-siglevel)[2],5),")", "\n")
  }
  
  t_values <- seq(0, 1, by = 0.01)
  
  roc_biweibull_df <- data.frame(
    FPR = t_values,
    TPR = t_values^((1 - mean(AUC_post)) / mean(AUC_post)),
    Method = "Bayesian bi-Weibull"
  )
  
  p <- ggplot(roc_biweibull_df, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1.2) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Bayesian bi-Weibull" = "darkolivegreen4")) +
    labs(title = "ROC Curves: bi-Weibull",
         x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) + 
    theme(
      legend.position = "none"
    )
  
  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}