# Function to generate Dirichlet weights using exponential distribution
generate_dirichlet_weights_bb <- function(size) {
  w <- rexp(size)  # Exponentially distributed with rate 1
  p <- w / sum(w)  # Normalize to create Dirichlet weights
  return(p)
}

# Step 1: Impute placement variables based on BB resampling distribution
impute_U_bb <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  
  # Generate Dirichlet weights for non-diseased group
  p <- generate_dirichlet_weights_bb(m)
  
  # Compute the survival function F# of the non-diseased group for each y_diseased
  U <- sapply(Y, function(yj) {
    1 - sum(p * (X <= yj))  # Placement variable U_j = 1 - F#(y_j)
  })
  
  return(U)
}

# Step 2: Generate one random realization of the ROC curve
generate_random_ROC_bb <- function(U, t_values) {
  n <- length(U)
  
  # Generate Dirichlet weights for diseased group (q)
  q <- generate_dirichlet_weights_bb(n)
  
  # Compute the ROC curve R(t) for each t in t_values
  ROC_t <- sapply(t_values, function(t) {
    sum(q * (U <= t))  # Weighted sum of indicators U_j <= t
  })
  
  return(ROC_t)
}

# Step 3: Averaging the ensemble of random ROC curves
ROC_realization_bb <- function(X, Y, t_values, replications = 1000) {
  n <- length(t_values)
  ROC_matrix <- matrix(0, nrow = replications, ncol = n)
  
  # Perform multiple BB replications
  for (r in 1:replications) {
    U <- impute_U_bb(X, Y)  # Step 1: Impute U
    ROC_matrix[r, ] <- generate_random_ROC_bb(U, t_values)  # Step 2: Generate ROC
  }
  
  # Step 3: Average over all replications to get the BB estimate of ROC
  ROC_realiz <- as.data.frame(ROC_matrix)
  
  return(ROC_realiz)
}

# AUC calculation using Simpson's rule
calculate_AUC_bb <- function(ROC_curve, t_values) {
  n <- length(t_values)
  
  # Simpson's rule requires an odd number of points
  if ((n - 1) %% 2 != 0) {
    stop("Simpson's rule requires an even number of intervals (odd number of points)")
  }
  
  h <- diff(t_values)[1]  # Assumes equal spacing in t_values
  
  # Simpson's rule: (h/3) * [f(x_0) + 4 * f(x_1) + 2 * f(x_2) + 4 * f(x_3) + ... + f(x_n)]
  auc_value <- (h / 3) * (ROC_curve[1] + 
                            4 * sum(ROC_curve[seq(2, n - 1, by = 2)]) + 
                            2 * sum(ROC_curve[seq(3, n - 2, by = 2)]) + 
                            ROC_curve[n])
  
  return(auc_value)
}

credibleble_interval_bb <- function(AUC_k, alpha=0.05){
  lower_bound <- quantile(AUC_k, probs = alpha / 2)
  upper_bound <- quantile(AUC_k, probs = 1 - alpha / 2)
  return(c(lower_bound, upper_bound))
}
