
binormal <- function(X,Y,ci=TRUE,ci_method="bootstrap",siglevel=0.05,n_bootstrap = 1000){
  set.seed(1691)
  params_X <- estimate_normal(X)
  params_Y <- estimate_normal(Y)
  
  mu1 <- params_X$mu
  sigma1 <- params_X$sigma
  mu2 <- params_Y$mu
  sigma2 <- params_Y$sigma
  
  # Compute binormal AUC
  auc_norm <- pnorm((mu2 - mu1) / sqrt(sigma1^2 + sigma2^2))
  
  ci_norm_boot <- bootstrap_auc_ci_binormal(X, Y, mu1, mu2, sigma1, sigma2, n_bootstrap, 1-siglevel)
  ci_mle <- auc_ci_mle_delta_binormal(mu1, sigma1, length(X), mu2, sigma2, length(Y),1-siglevel)
  
  msg <- paste0("AUC:",round(auc_norm,5),"\n")
  if(ci){
    if(ci_method == "bootstrap" | ci_method == "all"){
      msg <- paste0(msg,(1-siglevel)*100,"% Bootstrap CI:(", round(ci_norm_boot[1],5),",",round(ci_norm_boot[2],5),")","\n")
    }
    if(ci_method == "mle" | ci_method == "all"){
      msg <- paste0(msg,(1-siglevel)*100, "% MLE CI:(", round(ci_mle[1],5),",",round(ci_mle[2],5),")","\n")
    }
  }
  
  a <- (mu1 - mu2)/sigma2
  b <- sigma1/sigma2
  roc_param <- function(u) 1 - pnorm(a + b * qnorm(1 - u))

  u_grid <- seq(0, 1, length.out = 1000)
  tpr_param <- roc_param(u_grid)

  df_binormal <- data.frame(
    FPR = u_grid,
    TPR = tpr_param, 
    Method = "Binormal"
  )
  p <- ggplot(df_binormal, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Binormal" = "#00BA37")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) + 
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )
  
  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

bootstrap_auc_ci_binormal <- function(X, Y, mu1, mu2, sigma1, sigma2, n_bootstrap = 1000, conf_level = 0.95) {
  aucs <- numeric(n_bootstrap)
  n_cores <- parallel::detectCores() - 1  # Leave 1 core free
  cl <- makeCluster(n_cores)
  registerDoParallel(cl)
  clusterExport(cl, varlist = c("estimate_normal","fitdistr"), envir = environment())
  # Run the bootstrap loop in parallel
  aucs <- foreach(i = 1:n_bootstrap, .combine = c) %dopar% {
    m<- length(X)
    n <- length(Y)
    X_boot <- rnorm(m, mean = mu1, sd = sigma1)
    Y_boot <- rnorm(n, mean = mu2, sd = sigma2)
    
    params_Xb <- estimate_normal(X_boot)
    params_Yb <- estimate_normal(Y_boot)
    
    mu1b <- params_Xb$mu
    sigma1b <- params_Xb$sigma
    mu2b <- params_Yb$mu
    sigma2b <- params_Yb$sigma
    
    pnorm((mu2b - mu1b) / sqrt(sigma1b^2 + sigma2b^2))
  }
  
  suppressWarnings({
    stopCluster(cl)
  })
  
  # Calculate confidence interval
  lower_bound <- quantile(aucs, (1 - conf_level) / 2)
  upper_bound <- quantile(aucs, 1 - (1 - conf_level) / 2)
  
  # Corrected return statement to return a vector
  return(c(lower_bound, upper_bound))
}

auc_ci_mle_delta_binormal <- function(mu1, sigma1, m, mu2, sigma2, n, conf_level = 0.95) {
  S   <- sqrt(sigma1^2 + sigma2^2)
  d   <- (mu2 - mu1) / S
  auc <- pnorm(d)
  phi <- dnorm(d)
  
  # Partials derivtives
  dd_mu1    <- -1 / S
  dd_mu2    <-  1 / S
  dd_sigma1 <- - d * sigma1 / (S^2)
  dd_sigma2 <- - d * sigma2 / (S^2)
  
  # Asymptotic variances of MLEs 
  var_mu1    <- sigma1^2 / m
  var_mu2    <- sigma2^2 / n
  var_sigma1 <- sigma1^2 / (2 * m)
  var_sigma2 <- sigma2^2 / (2 * n)
  
  # Variance via delta method
  var_d   <- dd_mu1^2 * var_mu1 + dd_mu2^2 * var_mu2 +
    dd_sigma1^2 * var_sigma1 + dd_sigma2^2 * var_sigma2
  se_auc  <- abs(phi) * sqrt(var_d)
  
  z <- qnorm(1 - (1 - conf_level) / 2)
  ci <- auc + c(-1, 1) * z * se_auc
  ci <- pmax(0, pmin(1, ci))  # keep within [0,1]
  return(ci)
}


# Function to estimate mean and SD via MLE
estimate_normal <- function(data) {
  fit <- fitdistr(data, densfun = "normal")
  mu <- fit$estimate[1]
  sigma <- fit$estimate[2]
  return(list(mu = mu, sigma = sigma))
}