
bootstrap_auc_ci_biweibull <- function(X, Y,conf_level = 0.95, n_bootstrap = 1000) {
  m<- length(X)
  n <- length(Y)
  para_est <- estimate_param_biweibull(X, Y)
  a_hat <- para_est$alpha
  s0_hat <- para_est$s0  # s0 is theta_0
  s1_hat <- para_est$s1  # s1 is theta_1
  aucs <- numeric(n_bootstrap)
  n_cores <- parallel::detectCores() - 1  # Leave 1 core free
  cl <- makeCluster(n_cores)
  registerDoParallel(cl)
  clusterExport(cl, varlist = c("estimate_param_biweibull","nleqslv"), envir = environment())
  # Run the bootstrap loop in parallel
  aucs <- foreach(i = 1:n_bootstrap, .combine = c) %dopar% {
    # Simulate data
    Xb <- rweibull(m, shape = a_hat, scale = s0_hat^(1 / a_hat))
    Yb <- rweibull(n, shape = a_hat, scale = s1_hat^(1 / a_hat))
    para_est <- estimate_param_biweibull(Xb, Yb)
    ab_hat <- para_est$alpha
    s0b_hat <- para_est$s0
    s1b_hat <- para_est$s1
    s1b_hat / (s0b_hat + s1b_hat)
  }  
  
  suppressWarnings({
    stopCluster(cl)
  })
  # Calculate confidence interval
  lower_bound <- quantile(aucs, (1 - conf_level) / 2)
  upper_bound <- quantile(aucs, 1 - (1 - conf_level) / 2)
  
  # Corrected return statement to return a vector
  return(c(lower_bound, upper_bound))
}

# Calculate the confidence interval for the AUC estimate
auc_ci_mle_weibull<- function(auc_hat,m,n,a_hat, s0_hat, s1_hat, conf_level = 0.95) {
  # Constants used in the calculations
  eul_cons <- 0.5772156649  # Euler'-Mascheroni's constant
  gam_1 <- 1 - eul_cons     
  gam_2 <- -2 * eul_cons + eul_cons^2 + (pi^2) / 6  
  
  # Compute the Fisher information matrix components
  I_11 <- (-1) * (-1/a_hat^2) * ((m + n) * (1 + gam_2) + 2 * gam_1 * (n * log(s1_hat) + m * log(s0_hat)) + n * (log(s1_hat))^2 + m * (log(s0_hat))^2)
  I_22 <- (-1) * (-n) / s1_hat^2
  I_33 <- (-1) * (-m) / s0_hat^2
  I_12 <- (-1) * (n / (a_hat * s1_hat)) * (log(s1_hat) + gam_1)
  I_21 <- I_12
  I_13 <- (-1) * (m / (a_hat * s0_hat)) * (log(s0_hat) + gam_1)
  I_31 <- I_13
  I_23 <- 0
  I_32 <- 0
  
  # Normalize the Fisher information matrix components
  a_11 <- I_11 / m
  a_22 <- I_22 / n
  a_33 <- I_33 / n
  a_12 <- I_12 / (sqrt(m * n))
  a_21 <- a_12
  a_13 <- I_13 / (sqrt(m * n))
  a_31 <- a_13
  a_23 <- 0
  a_32 <- 0
  
  # Compute the determinant of the normalized Fisher information matrix
  u <- a_11 * a_22 * a_33 - a_12 * a_21 * a_33 - a_13 * a_31 * a_22
  
  # Compute the constant 'c' used in the confidence interval calculation
  c <- (s1_hat^2 * s0_hat^2) / ((s1_hat + s0_hat)^4)
  
  # Construct the matrix G=adj(A) for the confidence interval calculation
  G <- matrix(c(a_22 * a_33, -a_21 * a_33, -a_22 * a_31, 
                -a_21 * a_13, a_11 * a_33 - (a_13)^2, a_12 * a_31,
                -a_22 * a_31, a_12 * a_31, a_11 * a_22 - (a_12)^2), 
              nrow = 3, ncol = 3, byrow = TRUE)
  
  # Construct the vector 'b' for the confidence interval calculation
  b <- matrix(c(0, 1 / s1_hat, -1 / s0_hat), nrow = 3, ncol = 1, byrow = TRUE)
  
  # Compute the variance of the AUC estimate
  B <- (c / u) * (t(b) %*% G) %*% b
  
  # Calculate the lower and upper bounds of the 95% confidence interval for AUC
  Conf_AUE_lower <- auc_hat - qnorm(0.975) * sqrt(B / m)
  Conf_AUE_upper <- auc_hat + qnorm(0.975) * sqrt(B / m)
  
  return(c(Conf_AUE_lower, Conf_AUE_upper))
}

# Define function to estimate alpha
estimate_param_biweibull <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  objective_function <- function(a) {
    (m + n) / a + sum(log(Y)) + sum(log(X)) - 
      sum((Y^a) * log(Y)) / mean(Y^a) - 
      sum((X^a) * log(X)) / mean(X^a)
  }
  result <- nleqslv(x = 1, fn = objective_function)
  # Estimate shape parameter
  a_hat <- result$x
  # Estimate scale components
  s0_hat <- mean(X^a_hat)
  s1_hat <- mean(Y^a_hat)
  list(alpha=a_hat, s0=s0_hat, s1=s1_hat)
}