
calc_auc_empirical <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  
  auc <- sum(outer(Y, X, ">")) / (m * n) + 0.5 * sum(outer(Y, X, "==")) / (m * n)
  
  return(auc)
}

# Function to calculate bootstrap confidence interval for AUC
bootstrap_auc_ci_empirical <- function(X, Y, num_bootstrap = 1000, conf_level = 0.95) {
  # Initialize a vector to store the AUC values from each bootstrap sample
  auc_values <- numeric(num_bootstrap)
  
  n_cores <- parallel::detectCores() - 1  # Leave 1 core free
  cl <- makeCluster(n_cores)
  on.exit(stopCluster(cl))
  registerDoParallel(cl)
  clusterExport(cl, varlist = c("roc_empirical", "calc_auc_empirical"), envir = environment())
  # Run the bootstrap loop in parallel
  auc_values <- foreach(i = 1:num_bootstrap, .combine = c) %dopar% {
    # Resample with replacement from X and Y
    X_boot <- sample(X, length(X), replace = TRUE)
    Y_boot <- sample(Y, length(Y), replace = TRUE)
    
    # Calculate AUC for the bootstrap sample
    auc_values[i] <- calc_auc_empirical(X_boot, Y_boot)
  }
  
  # Calculate the confidence interval
  lower_bound <- quantile(auc_values, (1 - conf_level) / 2)
  upper_bound <- quantile(auc_values, 1 - (1 - conf_level) / 2)
  
  return(c(lower_bound, upper_bound))
}

# Function to calculate Q1 and Q2
calc_q1_q2_empirical <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  
  # Calculate Q1: Probability that two Ys exceed X
  q1 <- 0
  for (i in 1:(n - 1)) {
    for (j in (i + 1):n) {
      q1 <- q1 + sum(Y[i] > X & Y[j] > X)
    }
  }
  q1 <- q1 / (choose(n, 2) * m)
  
  # Calculate Q2: Probability that Y exceeds two Xs
  q2 <- 0
  for (i in 1:(m - 1)) {
    for (j in (i + 1):m) {
      q2 <- q2 + sum(Y > X[i] & Y > X[j])
    }
  }
  q2 <- q2 / (choose(m, 2) * n)
  
  return(list(Q1 = q1, Q2 = q2))
}

# Function to calculate variance of empirical AUC using Hanley's method
calc_var_auc_empirical <- function(auc, X, Y) {
  m <- length(X)
  n <- length(Y)
  
  # Calculate Q1 and Q2
  q_values <- calc_q1_q2_empirical(X, Y)
  Q1 <- q_values$Q1
  Q2 <- q_values$Q2
  
  # Variance of empirical AUC
  var_auc <- (auc * (1 - auc) + (n - 1) * (Q1 - auc^2) + (m - 1) * (Q2 - auc^2)) / (m * n)
  
  return(var_auc)
}

# Function to calculate confidence interval for AUC
calc_ci_auc_hm_empirical <- function(auc, var_auc, alpha = 0.05) {
  z_alpha <- qnorm(1 - alpha / 2)
  se_auc <- sqrt(var_auc)
  
  lower_ci <- max((auc - z_alpha * se_auc), 0)
  upper_ci <- min((auc + z_alpha * se_auc), 1)
  
  return(c(lower_ci, upper_ci))
}

# Function to calculate placement values
calc_placement_values_empirical <- function(X, Y) {
  
  # Placement values for X (with respect to Y)
  S_D_X <- sapply(X, function(xi) {
    mean(Y > xi) + mean(0.5*(Y==xi)) # Proportion of values in Y that are greater than xi
  })
  
  # Placement values for Y (with respect to X)
  S_barD_Y <- sapply(Y, function(yi) {
    mean(X > yi) + mean(0.5*(X==yi))  # Proportion of values in X that are greater than yi
  })
  
  return(list(S_D_X = S_D_X, S_barD_Y = S_barD_Y))
}

# Function to calculate the variance of the AUC using DeLong's method
calc_var_auc_delong_empirical <- function(X, Y) {
  m <- length(X)  # Sample size of population \bar{D}
  n <- length(Y)  # Sample size of population D
  
  # Get the placement values
  placement_values <- calc_placement_values_empirical(X, Y)
  S_D_X <- placement_values$S_D_X
  S_barD_Y <- placement_values$S_barD_Y
  
  # Variance of the empirical AUC based on DeLong's method
  var_auc <- (var(S_D_X) / n) + (var(S_barD_Y) / m)
  
  return(var_auc)
}

# Function to calculate confidence interval for AUC
calc_ci_auc_delong_empirical <- function(auc, var_auc, alpha = 0.05) {
  z_alpha <- qnorm(1 - alpha / 2)
  se_auc <- sqrt(var_auc)
  
  lower_ci <- max((auc - z_alpha * se_auc), 0)
  upper_ci <- min((auc + z_alpha * se_auc), 1)
  
  return(c(lower_ci, upper_ci))
}

roc_empirical <- function(X, Y) {
  x_val <- unique(sort(c(X, Y)))
  thresholds <- c(x_val[1] - 0.005, x_val, x_val[length(x_val)] + 0.005)
  FPR <- 1 - sapply(thresholds, function(t) empirical_cdf(X, t))
  TPR <- 1 - sapply(thresholds, function(t) empirical_cdf(Y, t))
  data.frame(FPR = FPR, TPR = TPR, Method = "Empirical")
}

empirical_cdf <- function(data, t) mean(data <= t)