
gamma_weibull <- function(X, Y, ci=TRUE, alpha=0.05, nboot=500){
  set.seed(1691)
  params_X <- estimate_gamma_fitdistr(X)
  params_Y <- estimate_gamma_fitdistr(Y)
  
  k1 <- params_X$k
  theta1 <- params_X$theta
  k2 <- params_Y$k
  theta2 <- params_Y$theta
  
  t_beta <- theta2 / (theta1 + theta2)
  auc_gamma_beta <- pbeta(t_beta, shape1 = k1, shape2 = k2)
  #print(auc_gamma_beta)
  
  #f_value <- (k2 * theta2) / (k1 * theta1)
  #auc_gamma_f <- pf(f_value, df1 = 2*k1, df2 = 2*k2)
  #print(auc_gamma_f)
  
  # Bootstrap
  aucs <- numeric(nboot)
  
  for (i in 1:nboot) {
    X_boot <- rgamma(length(X), shape = k1, rate = 1/theta1)
    Y_boot <- rgamma(length(Y), shape = k2, rate = 1/theta2)
    
    params_Xb <- estimate_gamma_fitdistr(X_boot)
    params_Yb <- estimate_gamma_fitdistr(Y_boot)
    
    k1b <- params_Xb$k
    theta1b <- params_Xb$theta
    k2b <- params_Yb$k
    theta2b <- params_Yb$theta
    
    t_b <- theta2b / (theta1b + theta2b)
    aucs[i] <- pbeta(t_b, shape1 = k1b, shape2 = k2b)
  }

  ci_lower_vec <- quantile(aucs, probs = alpha / 2)
  ci_upper_vec <- quantile(aucs, probs = 1 - alpha / 2)
  
  msg <- "Summary:\n"
  msg <- paste0("AUC :", round(auc_gamma_beta,5), "\n")
  
  if(ci){
    msg <- paste0(msg,(1-alpha)*100, "% CI :(", round(ci_lower_vec,5),",",round(ci_upper_vec,5),")", "\n")
  }
  
  thresholds <- seq(min(c(X, Y)), max(c(X, Y)), length.out = 200)
  FPR_gamma <- 1 - pgamma(thresholds, shape = params_X$k, scale = params_X$theta)
  TPR_gamma <- 1 - pgamma(thresholds, shape = params_Y$k, scale = params_Y$theta)
  
  df_bigamma <- data.frame(FPR = FPR_gamma, TPR = TPR_gamma, Method = "Bi-Gamma")
  
  p <- ggplot(df_bigamma, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Bi-Gamma" = "#4169E1")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) + 
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )
  
  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}

estimate_gamma_fitdistr <- function(data) {
  fit <- fitdistr(data, densfun = "gamma")
  shape <- fit$estimate[1]   # shape parameter k
  rate <- fit$estimate[2]    # rate = 1/scale
  theta <- 1 / rate          # scale = 1/rate
  return(list(k = shape, theta = theta))
}

