

# Load the necessary package for bandwidth selection
set.seed(1691)

# Kernel density estimator function using specified kernel (biweight)
kde_norm_norm_ucv <- function(x, sample, h) {
  m <- length(sample)
  kde_val <- numeric(length(x))
  for (i in 1:m) {
    kde_val <- kde_val + pnorm((x - sample[i]) / h)
  }
  return(kde_val / m)
  
}


# Define ROC function with Kernel Density Estimation
ROC_kde_normal_norm_ucv <- function(X, Y, h_x, h_y) {
  # Define a fine grid over which to calculate the ROC curve
  x_vals <- seq(min(c(X, Y))-3*h_x, max(c(X, Y))+3*h_y , length.out = 1000)
  F_hat <- kde_norm_norm_ucv(x_vals, X, h_x)
  G_hat <- kde_norm_norm_ucv(x_vals, Y, h_y)
  
  # Step 3: Compute cumulative distributions F_hat and G_hat
  FPR <- 1-F_hat  # Survival for non-diseased
  TPR <- 1-G_hat  # Survival for diseased
  # Return ROC points
  FPR <- c(0, 1, FPR)
  TPR <- c(0, 1, TPR)
  return(data.frame(FPR = FPR, TPR = TPR))
}

# Function to calculate AUC using the trapezoidal rule
calculate_auc_norm_ucv <- function(x, y) {
  sorted_indices <- order(x)
  x <- x[sorted_indices]
  y <- y[sorted_indices]
  auc <- sum((x[-1] - x[-length(x)]) * (y[-1] + y[-length(y)]) / 2)
  if (auc<0) {return(0)}
  if (auc>1) {return(1)}
  else (return(auc))
}

# Function to calculate empirical AUC
calc_auc_empirical_norm_ucv <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  auc <- sum(outer(Y, X, ">")) / (m * n) + 0.5 * sum(outer(Y, X, "==")) / (m * n)
  return(auc)
}

# Function to calculate placement values
calc_placement_values_norm_ucv <- function(X, Y) {
  S_D_X <- sapply(X, function(xi) {
    mean(Y > xi) + mean(0.5 * (Y == xi))
  })
  S_barD_Y <- sapply(Y, function(yi) {
    mean(X > yi) + mean(0.5 * (X == yi))
  })
  return(list(S_D_X = S_D_X, S_barD_Y = S_barD_Y))
}
# Function to calculate the variance of the AUC using DeLong's method
calc_var_auc_delong_norm_ucv <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  placement_values <- calc_placement_values_norm_ucv(X, Y)
  S_D_X <- placement_values$S_D_X
  S_barD_Y <- placement_values$S_barD_Y
  var_auc <- (var(S_D_X) / n) + (var(S_barD_Y) / m)
  return(var_auc)
}

# Function to calculate confidence interval for AUC
calc_ci_auc_delong_norm_ucv <- function(auc, var_auc, alpha = 0.05) {
  z_alpha <- qnorm(1 - alpha / 2)
  se_auc <- sqrt(var_auc)
  if (auc == 1) {
    lower_ci <- max((auc - z_alpha * se_auc), 0)
    upper_ci <- min((auc + z_alpha * se_auc), 1)
    return(c(lower_ci, upper_ci))
  } else if (auc <1) {
    frac <- se_auc / (1 - auc)
    lower_ci <- 1 - (1 - auc) * exp(z_alpha * frac)
    upper_ci <- 1 - (1 - auc) * exp(-z_alpha * frac)
    return(c(max(0, lower_ci), min(1, upper_ci)))
    #return(c(lower_ci, upper_ci))
  }
}

# Fix in bootstrap_auc_ci function return statement
bootstrap_auc_ci_norm_ucv <- function(X, Y, n_bootstrap = 1000, conf_level = 0.95) {
  auc_values <- numeric(n_bootstrap)
  
  # Perform bootstrap resampling
  for (i in 1:n_bootstrap) {
    
  }
  
  n_cores <- parallel::detectCores() - 1  # Leave 1 core free
  cl <- makeCluster(n_cores)
  registerDoParallel(cl)
  clusterExport(cl, varlist = c("h.ucv", "ROC_kde_normal_norm_ucv", "calculate_auc_norm_ucv","kde_norm_norm_ucv"), envir = environment())
  # Run the bootstrap loop in parallel
  auc_values <- foreach(i = 1:n_bootstrap, .combine = c) %dopar% {
    boot_X <- sample(X, length(X), replace = TRUE)
    boot_Y <- sample(Y, length(Y), replace = TRUE)
    h_xx <- h.ucv(boot_X)$h
    h_yy <- h.ucv(boot_Y)$h
    roc_data <- ROC_kde_normal_norm_ucv(boot_X, boot_Y, h_xx, h_yy)
    calculate_auc_norm_ucv(roc_data$FPR, roc_data$TPR)
  }
  
  suppressWarnings({
    stopCluster(cl)
  })
  
  # Calculate confidence interval
  lower_bound <- quantile(auc_values, (1 - conf_level) / 2)
  upper_bound <- quantile(auc_values, 1 - (1 - conf_level) / 2)
  
  # Corrected return statement to return a vector
  return(c(lower_bound, upper_bound))
}



