
lehmann <- function(X, Y, ci=TRUE,ci_method="ple",siglevel=0.05){
  # Combine the scores and classes for Cox model analysis
  score <- c(X, Y)  # Combine scores from both groups
  class <- c(rep(0, length(X)), rep(1, length(Y)))  # Class labels (0: not recovered, 1: recovered)
  status <- rep(1, length(X) + length(Y))  # all observations are complete)
  
  # Create a data frame for Cox model analysis
  df_cox <- data.frame(score, status, class)
  
  # Fit the Cox proportional hazards model
  res <- coxph(Surv(score, status) ~ class, data = df_cox)
  
  # Summarize the Cox model results
  res.summary <- summary(res)
  
  # Extract the coefficient (log hazard ratio) from the Cox model
  b_hat <- res.summary$coefficients[, "coef"]
  
  # Calculate the AUC estimate using the Cox model
  auc_hat_cox <- 1 / (1 + exp(b_hat))  # PL estimate of AUC
  
  
  # Calculate the standard error for the Cox model AUC estimate
  SE <- res.summary$coefficients[, "se(coef)"] * exp(b_hat) / ((1 + exp(b_hat))^2)
  
  # Calculate the lower and upper bounds of the 95% confidence interval for the Cox model AUC
  L_CI_cox <- 1 / (1 + exp(b_hat)) - SE * qnorm(1-siglevel/2)
  U_CI_cox <- 1 / (1 + exp(b_hat)) + SE * qnorm(1-siglevel/2)
  
  # Print the confidence interval for the Cox model AUC
  msg <- "Summary:\n"
  msg <- paste0(msg,"AUC estimate :", round(auc_hat_cox,5), "\n")
  if(ci){
    if(ci_method == "ple"){
      msg <- paste0(msg,(1-siglevel)*100, "% PLE CI :","(",round(L_CI_cox,5), " , ", round(U_CI_cox,5),")","\n")
    }
  }
  
  df_lehmann <- data.frame(
    FPR = seq(0, 1, length.out = 100),
    TPR = seq(0, 1, length.out = 100) ^ ((1 - auc_hat_cox) / auc_hat_cox),
    Method = "Lehmann"
  )
  
  p <- ggplot(df_lehmann, aes(x = FPR, y = TPR, color = Method)) +
    geom_line(linewidth = 1) +
    geom_abline(slope = 1, intercept = 0, linetype = "dashed", color = "red") +
    scale_color_manual(values = c("Lehmann" = "#008B8B")) +
    labs(x = "False Positive Rate (FPR)",
         y = "True Positive Rate (TPR)") +
    theme_bw(base_size = 14) + 
    theme(
      legend.position = "NONE",
      panel.grid = element_line(color = "white")
    )
  
  print(p)
  cat(msg)
  return(list(plot=p,summary=msg))
}