empirical_cdf_order <- function(data, t) {
  mean(data <= t)
}

# Function to compute P_mn(t) for combined samples
combined_cdf_order <- function(t, X, Y) {
  m <- length(X)
  n <- length(Y)
  F_m <- empirical_cdf_order(X, t)
  G_n <- empirical_cdf_order(Y, t)
  (m / (m + n)) * F_m + (n / (m + n)) * G_n
}

# Function to compute F_mn and G_mn under order restrictions

F_mn_restricted_order <- function(t, X, Y) {
  F_m <- empirical_cdf_order(X, t)
  G_n <- empirical_cdf_order(Y, t)
  P_mn <- combined_cdf_order(t, X, Y)
  return(pmax(F_m, P_mn))
}

G_mn_restricted_order <- function(t, X, Y) {
  F_m <- empirical_cdf_order(X, t)
  G_n <- empirical_cdf_order(Y, t)
  P_mn <- combined_cdf_order(t, X, Y)
  return(pmin(G_n, P_mn))
}

# Function to calculate the order-restricted ROC curve
roc_order <- function(X, Y) {
  # Define a range of thresholds
  x_val <- unique(sort(X))
  thresholds <- x_val
  FPR <- 1 - sapply(thresholds, function(t) F_mn_restricted_order(t, X, Y))  # 1 - F_mn
  TPR <- 1 - sapply(thresholds, function(t) G_mn_restricted_order(t, X, Y))  # 1 - G_mn
  # Return ROC points
  FPR <- c(0, FPR, 1)
  TPR <- c(0, TPR, 1)
  data.frame(FPR = FPR, TPR = TPR, Method = "Order")
}


# Compute ROC curve points under order restriction
#roc_data_order <- roc_curve_order_restriction(non_diseased, diseased)

# Function to calculate AUC using the trapezoidal rule
calculate_auc_order <- function(x, y) {
  # Sort points by FPR to ensure the trapezoidal rule applies correctly
  sorted_indices <- order(x)
  x <- x[sorted_indices]
  y <- y[sorted_indices]
  sm <- sum((x[-1] - x[-length(x)]) * (y[-1] + y[-length(y)]) / 2)
  return(sm)
}

# Function to calculate empirical AUC
calc_auc_empirical_order <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  
  auc <- sum(outer(Y, X, ">")) / (m * n) + 0.5 * sum(outer(Y, X, "==")) / (m * n)
  
  return(auc)
}

# Function to calculate Q1 and Q2
calc_q1_q2_order <- function(X, Y) {
  m <- length(X)
  n <- length(Y)
  
  # Calculate Q1: Probability that two Ys exceed X
  q1 <- 0
  for (i in 1:(n - 1)) {
    for (j in (i + 1):n) {
      q1 <- q1 + sum(Y[i] > X & Y[j] > X)
    }
  }
  q1 <- q1 / (choose(n, 2) * m)
  
  # Calculate Q2: Probability that Y exceeds two Xs
  q2 <- 0
  for (i in 1:(m - 1)) {
    for (j in (i + 1):m) {
      q2 <- q2 + sum(Y > X[i] & Y > X[j])
    }
  }
  q2 <- q2 / (choose(m, 2) * n)
  
  return(list(Q1 = q1, Q2 = q2))
}

# Function to calculate variance of empirical AUC using Hanley's method
calc_var_auc_empirical_order <- function(auc, X, Y) {
  m <- length(X)
  n <- length(Y)
  
  # Calculate Q1 and Q2
  q_values <- calc_q1_q2_order(X, Y)
  Q1 <- q_values$Q1
  Q2 <- q_values$Q2
  
  # Variance of empirical AUC
  var_auc <- (auc * (1 - auc) + (n - 1) * (Q1 - auc^2) + (m - 1) * (Q2 - auc^2)) / (m * n)
  
  return(var_auc)
}

# Function to calculate confidence interval for AUC
calc_ci_auc_hm_order <- function(auc, var_auc, alpha = 0.05) {
  z_alpha <- qnorm(1 - alpha / 2)
  se_auc <- sqrt(var_auc)
  
  lower_ci <- max((auc - z_alpha * se_auc), 0)
  upper_ci <- min((auc + z_alpha * se_auc), 1)
  
  return(c(lower_ci, upper_ci))
}

# Function to calculate placement values
calc_placement_values_order <- function(X, Y) {
  
  # Placement values for X (with respect to Y)
  S_D_X <- sapply(X, function(xi) {
    mean(Y > xi) + mean(0.5*(Y==xi)) # Proportion of values in Y that are greater than xi
  })
  
  # Placement values for Y (with respect to X)
  S_barD_Y <- sapply(Y, function(yi) {
    mean(X > yi) + mean(0.5*(X==yi))  # Proportion of values in X that are greater than yi
  })
  
  return(list(S_D_X = S_D_X, S_barD_Y = S_barD_Y))
}

# Function to calculate the variance of the AUC using DeLong's method
calc_var_auc_delong_order <- function(X, Y) {
  m <- length(X)  # Sample size of population \bar{D}
  n <- length(Y)  # Sample size of population D
  
  # Get the placement values
  placement_values <- calc_placement_values_order(X, Y)
  S_D_X <- placement_values$S_D_X
  S_barD_Y <- placement_values$S_barD_Y
  
  # Variance of the empirical AUC based on DeLong's method
  var_auc <- (var(S_D_X) / n) + (var(S_barD_Y) / m)
  
  return(var_auc)
}

# Function to calculate confidence interval for AUC
calc_ci_auc_delong_order <- function(auc, var_auc, alpha = 0.05) {
  z_alpha <- qnorm(1 - alpha / 2)
  se_auc <- sqrt(var_auc)
  
  lower_ci <- max((auc - z_alpha * se_auc), 0)
  upper_ci <- min((auc + z_alpha * se_auc), 1)
  
  return(c(lower_ci, upper_ci))
}

# Fix in bootstrap_auc_ci function return statement
bootstrap_auc_ci_order <- function(X, Y, n_bootstrap = 1000, conf_level = 0.95) {
  auc_values <- numeric(n_bootstrap)
  
  n_cores <- parallel::detectCores() - 1  # Leave 1 core free
  cl <- makeCluster(n_cores)
  on.exit(stopCluster(cl))
  registerDoParallel(cl)
  clusterExport(cl, varlist = c("roc_order", "calculate_auc_order","F_mn_restricted_order","G_mn_restricted_order","combined_cdf_order","empirical_cdf_order"), envir = environment())
  # Run the bootstrap loop in parallel
  auc_values <- foreach(i = 1:n_bootstrap, .combine = c) %dopar% {
    boot_X <- sample(X, length(X), replace = TRUE)
    boot_Y <- sample(Y, length(Y), replace = TRUE)
    roc_data <- roc_order(boot_X, boot_Y)
    calculate_auc_order(roc_data$FPR, roc_data$TPR)
  }
  
  # suppressWarnings({
  #   stopCluster(cl)
  # })
  
  # Calculate confidence interval
  lower_bound <- quantile(auc_values, (1 - conf_level) / 2)
  upper_bound <- quantile(auc_values, 1 - (1 - conf_level) / 2)
  
  # Corrected return statement to return a vector
  return(c(lower_bound, upper_bound))
}

