#' @noRd
K_CDF <- function(u, kernel = c("gaussian", "biweight", "epanechnikov")) {
  kernel <- match.arg(kernel)
  vapply(u, function(x) {
    if (kernel == "gaussian") return(pnorm(x))
    if (kernel == "biweight") {
      if (x < -1) return(0)
      if (x > 1) return(1)
      return((15/16) * (x - (2/3)*x^3 + (1/5)*x^5 + 8/15))
    }
    if (x < -1) return(0)
    if (x > 1) return(1)
    (3/4) * (x - (1/3)*x^3 + 2/3)
  }, numeric(1))
}

#' @noRd
#' @references
#' Altman, N., & Leger, C. (1995). Bandwidth selection for kernel distribution function estimation. \emph{Journal of Statistical Planning and Inference}, 46(2), 195-214.
bandwidth_AL <- function(X, kernel = c("gaussian", "biweight", "epanechnikov")) {
  kernel <- match.arg(kernel)
  m <- length(X)
  phi <- dnorm; M <- function(u) (u^2 - 1) * phi(u)
  D <- outer(X, X, "-")
  g <- min(sd(X), IQR(X)/1.349) * m^(-0.2)
  U_g <- D / g
  psi2_hat <- (1 / m^2) * (1/g^3) * sum(M(U_g))
  eps <- .Machine$double.eps
  rho_k <- switch(kernel, gaussian = 1/sqrt(pi), biweight = 50/231, epanechnikov = 9/35)
  mu2_k <- switch(kernel, gaussian = 1, biweight = 1/7, epanechnikov = 1/5)
  denom <- m * (mu2_k^2) * max(eps, -psi2_hat)
  (rho_k / denom)^(1/3)
}

#' @noRd
#' @references
#' Polansky, A. M., & Baker, E. R. (2000). Multistage plug-in bandwidth selection for kernel distribution function estimates. \emph{Journal of Statistical Computation and Simulation}, 65(1-4), 63-80.
bandwidth_PB <- function(X, kernel = c("gaussian", "biweight", "epanechnikov")) {
  kernel <- match.arg(kernel)
  m <- length(X)
  phi <- dnorm; M <- function(u) (u^2 - 1) * phi(u); L <- function(u) (u^4 - 6*u^2 + 3) * phi(u)
  rho_k <- switch(kernel, gaussian = 1/sqrt(pi), biweight = 50/231, epanechnikov = 9/35)
  mu2_k <- switch(kernel, gaussian = 1, biweight = 1/7, epanechnikov = 1/5)
  D <- outer(X, X, "-")
  sigma_hat <- min(sd(X), IQR(X)/1.349)
  d_hat <- ((16 * sqrt(2) / (5 * m))^(1/7)) * sigma_hat
  U_d <- D / d_hat
  psi_d_hat <- (1 / m^2) * (1 / d_hat^5) * sum(L(U_d))
  eps <- .Machine$double.eps
  q_hat <- (sqrt(2) / (m * sqrt(pi) * max(psi_d_hat, eps)))^(1/5)
  U_q <- D / q_hat
  psi2_hat <- (1 / m^2) * (1 / q_hat^3) * sum(M(U_q))
  denom <- m * (mu2_k^2) * max(eps, -psi2_hat)
  (rho_k / denom)^(1/3)
}

#' @noRd
bandwidth_BHP <- function(X, kernel = c("gaussian", "biweight", "epanechnikov")) {
  kernel <- match.arg(kernel)
  m <- length(X)
  sigma_hat <- min(sd(X), IQR(X)/1.349)
  rho_k <- switch(kernel, gaussian = 1/sqrt(pi), biweight = 50/231, epanechnikov = 9/35)
  mu2_k <- switch(kernel, gaussian = 1, biweight = 1/7, epanechnikov = 1/5)
  C_k <- (4 * sqrt(pi) * rho_k) / (mu2_k^2)
  C_k^(1/3) * sigma_hat * m^(-1/3)
}

#' @noRd
bandwidth_AR <- function(X, kernel = c("gaussian", "biweight", "epanechnikov")) {
  m <- length(X)
  sigma_hat <- min(sd(X), IQR(X)/1.349)
  1.4287 * sigma_hat * m^(-1/3)
}

#' @noRd
#' @references
#' Silverman, B. W. (1986). \emph{Density Estimation for Statistics and Data Analysis}. Chapman and Hall.
bandwidth_pdf <- function(X, kernel = c("gaussian", "biweight", "epanechnikov")) {
  m <- length(X)
  sigma_hat <- min(sd(X), IQR(X)/1.349)
  0.9 * sigma_hat * m^(-0.2)
}

#' @noRd
bandwidth <- function(X, method = c("pdf", "AL", "PB", "BHP", "AR"),
                      kernel = c("gaussian", "biweight", "epanechnikov")) {
  method <- match.arg(method); kernel <- match.arg(kernel)
  bw_val <- switch(method,
                   AL   = bandwidth_AL(X, kernel),
                   PB   = bandwidth_PB(X, kernel),
                   BHP  = bandwidth_BHP(X, kernel),
                   AR= bandwidth_AR(X, kernel),
                   pdf  = bandwidth_pdf(X, kernel))
  list(h_opt = bw_val, method = method, kernel = kernel)
}

#' @noRd
CDF_kde <- function(x, sample, h, kernel = c("gaussian", "biweight", "epanechnikov")) {
  kernel <- match.arg(kernel)
  sapply(x, function(xi) mean(K_CDF((xi - sample)/h, kernel)))
}

#' @noRd
#' @references
#' DeLong, E. R., DeLong, D. M., & Clarke-Pearson, D. L. (1988). Comparing the areas under two or more correlated receiver operating characteristic curves: a nonparametric approach. \emph{Biometrics}, 44(3), 837-845.
auc_kde_ci <- function(X, Y, h_m, h_n, kernel = c("gaussian", "biweight", "epanechnikov"), alpha = 0.05) {
  kernel <- match.arg(kernel)
  X <- as.numeric(X); Y <- as.numeric(Y)
  m <- length(X); n <- length(Y)
  D_YX <- outer(Y, X, function(y, x) (y - x) / h_m)
  U <- rowMeans(apply(D_YX, 2, K_CDF, kernel = kernel))
  D_XY <- outer(X, Y, function(x, y) (x - y) / h_n)
  V <- rowMeans(apply(D_XY, 2, K_CDF, kernel = kernel))
  auc <- mean(U)
  var_hat <- var(U)/n + var(V)/m
  se <- sqrt(max(var_hat, .Machine$double.eps))
  z <- qnorm(1 - alpha/2)
  ci <- c(max(0, auc - z*se), min(1, auc + z*se))
  list(auc_hat = auc, ci = ci)
}
