test_that("smoothROC checks biomarker is numeric", {
  data("dystrophy", package = "smoothROC")

  bad_data <- dystrophy
  bad_data$CK_chr <- as.character(bad_data$CK)

  expect_error(
    smoothROC(
      data      = bad_data,
      biomarker = "CK_chr",
      status    = "Class",
      diseased  = "carrier"
    ),
    "`biomarker` must be numeric",
    fixed = TRUE
  )
})

test_that("smoothROC checks status has exactly two levels", {
  data("dystrophy", package = "smoothROC")

  bad_data <- dystrophy
  bad_data$Class3 <- factor(ifelse(bad_data$Class == "carrier", "A", "B"))
  bad_data$Class3[1] <- "C"  # introduce a third level

  expect_error(
    smoothROC(
      data      = bad_data,
      biomarker = "CK",
      status    = "Class3",
      diseased  = "A"
    ),
    "`status` must have exactly two distinct values",
    fixed = TRUE
  )
})

test_that("smoothROC enforces positive biomarker for logtrans = TRUE", {
  data("dystrophy", package = "smoothROC")

  bad_data <- dystrophy
  bad_data$CK[1] <- 0

  expect_error(
    smoothROC(
      data      = bad_data,
      biomarker = "CK",
      status    = "Class",
      diseased  = "carrier",
      logtrans  = TRUE
    ),
    "Biomarker must be > 0 for log-transformation.",
    fixed = TRUE
  )
})

test_that("smoothROC rejects unsupported kernel and bw_method", {
  data("dystrophy", package = "smoothROC")

  # kernel argument is match.arg(), so invalid kernel should error
  expect_error(
    smoothROC(
      data      = dystrophy,
      biomarker = "CK",
      status    = "Class",
      diseased  = "carrier",
      kernel    = "invalid_kernel"
    ),
    "arg.*kernel",  # generic message from match.arg
    ignore.case = TRUE
  )

  # bw_method also uses match.arg
  expect_error(
    smoothROC(
      data      = dystrophy,
      biomarker = "CK",
      status    = "Class",
      diseased  = "carrier",
      bw_method = "invalid_bw"
    ),
    "arg.*bw_method",
    ignore.case = TRUE
  )
})

test_that("smoothROC requires at least two observations in each class", {
  data("dystrophy", package = "smoothROC")

  # make one group have only 1 observation
  small_data <- dystrophy[1:3, ]
  small_data$Class <- factor(c("carrier", "normal", "normal"))

  expect_error(
    smoothROC(
      data      = small_data,
      biomarker = "CK",
      status    = "Class",
      diseased  = "carrier"
    ),
    "Need >=2 observations in each class.",
    fixed = TRUE
  )
})

