test_that("smoothROC runs on dystrophy and returns a smoothROC object", {
  skip_on_cran()

  data("dystrophy", package = "smoothROC")

  roc <- smoothROC(
    data      = dystrophy,
    biomarker = "CK",
    status    = "Class",
    diseased  = "carrier",
    kernel    = "biweight",
    bw_method = "PB",
    alpha     = 0.05,
    logtrans  = TRUE,
    grid_n    = 2000
  )

  # Class and basic structure
  expect_s3_class(roc, "smoothROC")
  expect_true(is.list(roc))

  # Expected names (adjust if you change your return list)
  expect_true(all(c(
    "curve", "AUC", "AUC_ci_lo", "AUC_ci_hi", "AUC_ci",
    "J", "J_ci_lo", "J_ci_hi", "J_ci",
    "t0", "sensitivity", "specificity",
    "kernel", "bandwidth_method", "hX", "hY",
    "plot", "alpha"
  ) %in% names(roc)))

  # AUC and J within proper ranges
  expect_true(is.numeric(roc$AUC))
  expect_gte(roc$AUC, 0)
  expect_lte(roc$AUC, 1)

  expect_true(is.numeric(roc$J))
  expect_gte(roc$J, 0)
  expect_lte(roc$J, 1)

  # Confidence interval ordering
  expect_lte(roc$AUC_ci_lo, roc$AUC_ci_hi)
  expect_lte(roc$J_ci_lo,   roc$J_ci_hi)

  # curve data frame sanity checks
  expect_s3_class(roc$curve, "data.frame")
  expect_true(all(c("threshold", "FPR", "TPR", "J") %in% names(roc$curve)))

  # grid length
  expect_equal(nrow(roc$curve), 2000)

  # FPR and TPR in [0,1]
  expect_true(all(roc$curve$FPR >= 0 & roc$curve$FPR <= 1, na.rm = TRUE))
  expect_true(all(roc$curve$TPR >= 0 & roc$curve$TPR <= 1, na.rm = TRUE))

  # bandwidths positive
  expect_gt(roc$hX, 0)
  expect_gt(roc$hY, 0)
})

test_that("plot and print/summary methods for smoothROC work", {
  skip_on_cran()

  data("dystrophy", package = "smoothROC")

  roc <- smoothROC(
    data      = dystrophy,
    biomarker = "CK",
    status    = "Class",
    diseased  = "carrier",
    kernel    = "biweight",
    bw_method = "PB",
    alpha     = 0.05,
    logtrans  = TRUE,
    grid_n    = 200
  )

  # print / summary: no error
  expect_invisible(print(roc))
  expect_invisible(summary(roc))

  # plot with and without label: return ggplot object
  p1 <- plot(roc)
  p2 <- plot(roc, label = FALSE)

  expect_s3_class(p1, "ggplot")
  expect_s3_class(p2, "ggplot")
})
